/*
This uses the high level WAVE API PlaySound() to play some WAVE files. I use registry
keys to store the names of the wave files so that the user can reassign wave files
to my various registry keys using Control Panel's Sound utility. In other words, he
can customize the wave files that my app uses.

First, I need to add a registry key for my application under the tree
HKEY_CURRENT_USER\AppEvents\Schemes\Apps. I'll name this key "PlaySound_Tutor".
This is entirely arbitrary. You can name it whatever you want, subject to the 
limitations of registry key names (ie, characters ranging from values 32 through
127. no spaces nor a backslash (\) nor wildcard characters * or ?). Pick
some name that uniquely identifies your application. For example, if your application
is a WAVE file player, and your company name is "Blort Enterprises", perhaps your
key name will be "BlortWavePlayer".

Now, under this newly added key, I'll add the subkeys for my "Sound Actions". What's
that? A Sound Action is just action my program takes for which I want to play an
associated WAVE file. For example, maybe I want to play a WAVE file every time he
clicks the right mouse button upon a particular control. The action would be him
clicking on the button. I'm going to have 3 such Sound Actions in this example. Each
one will have its own subkey under my "PlaySound_Tutor" key. The first Sound Action
plays when the program starts. Therefore, the name that I choose for its registry key
is "ProgramStart". Again, this is arbitrary. The second Sound Action plays when the
program ends. Therefore, the name that I choose for its registry key is "ProgramEnd".
The third Sound Action I
intend to play as a sound effect whenever the user presses a key (other than the
space bar which ends the program). The name I choose for its registry key is
"SoundEffect1".

Under each of these subkeys, I need to add a .current subkey. It must be called that.
The default value for this subkey will be the name of the WAVE file I want assigned to
play for that Sound Action. For example, for the ProgramStart Sound Action, I'm going to
assign the WAVE file called "C:\\WINDOWS\\CHORD.WAV" to the .current subkey under
the ProgramStart registry key. Of course, in order for this example to work, you should
make sure that you have such a WAVE file on your system. In fact, I'm going to assign
the same WAVE file to the .current subkeys under all 3 Sound Event subkeys, although
normally you'd have different WAVE files for different actions.

Now, when the user brings up the Control Panel's Sound utility, he's going to see a
heading labeled "PlaySound_Tutor". Underneath this heading will be 3 labels of
"ProgramStart", "ProgramEnd" and "SoundEffect1". He can click on any one of those
3 labels and change the WAVE file associated with it. The Sound utility will modify
the .current subkey under the appropriate Sound Action subkey. For example, if he
changes ProgramStart to use the WAVE file named "D:\noise.wav", then the value for
the .current subkey under the ProgramStart subkey will be changed to that.

Perhaps you'd like for the Sound utility to display more meaningful labels to the user.
For example, instead of the label "ProgramStart", maybe you'd like the label to read
"When program starts" to give him an idea of the true purpose of that sound event. How
do you do this? Well, you merely set the default value for the ProgramStart subkey to
whatever you wish that label to read.

So too, you can change the "PlaySound_Tutor" label. Just set the default value for the
PlaySound_Tutor key to whatever you wish the label to be. In this example, I will set
it to "PlaySound Example".

You could use the functions deleteKey(), createAppKey(), createSoundAction(),
openSoundSchemeKey(), keyNameCopy(), and playSoundAction() verbatim in your program
to easily setup/play/delete your own Sound Actions. Just peruse the main() function
to see how they are used.
*/

#include <windows.h>
#include <stdio.h>
#include <conio.h>

#include <winreg.h>
#include <mmsystem.h>



/* Your application would change these to appropriate strings */
TCHAR		AppKeyName[] = "PlaySound_Tutor";
TCHAR		AppStrName[] = "PlaySound Example";
TCHAR		ProgramStart[] = "ProgramStart";
TCHAR		ProgramStartStr[] = "When program starts";
TCHAR		ProgramEnd[] = "ProgramEnd";
TCHAR		ProgramEndStr[] = "When program ends";
TCHAR		SoundEffect1[] = "SoundEffect1";
TCHAR		SoundEffect1Str[] = "Sound Effect 1";
TCHAR		WaveName[] = "C:\\WINDOWS\\CHORD.WAV";





/****************************** deleteKey() ******************************/
/* Deletes a key and all of its subkeys recursively.
 *
 * INPUTS:
 *		hStartKey -- Handle to key that contains the key we wish to delete.
 *		pKeyName -- Pointer to the null-terminated name of the key to delete.
 *
 * OUTPUTS;
 *		If successful, returns 0, otherwise non-zero.
 */

#define MAX_KEY_LENGTH 256
DWORD	SubKeyLength;

DWORD deleteKey(HKEY hStartKey, LPTSTR pKeyName)
{
	DWORD	dwRtn;
	TCHAR	SubKeyName[MAX_KEY_LENGTH];	/* This should be set to accomodate your longest key name */
	HKEY	hKey;

	if(!(dwRtn = RegOpenKeyEx(hStartKey, pKeyName, 0, KEY_ENUMERATE_SUB_KEYS | DELETE, &hKey)))
	{
		do
		{
			SubKeyLength = MAX_KEY_LENGTH - 1;

			if (!(dwRtn = RegEnumKeyEx(hKey, 0, &SubKeyName[0], &SubKeyLength, 0, 0, 0, 0)))
			{
				dwRtn = deleteKey(hKey, &SubKeyName[0]);
			}

			else if (dwRtn == ERROR_NO_MORE_ITEMS)
			{
				dwRtn = RegDeleteKey(hStartKey, pKeyName);
				break;
			}
		} while (!dwRtn);

		RegCloseKey(hKey);
	}

	return dwRtn;
}






/************************** openSoundSchemeKey() **************************/
/* This opens the HKEY_CURRENT_USER\AppEvents\Schemes\Apps key and returns
 * a handle to it, This is the registry key under which we add our App's
 * Sound Events.
 *
 * INPUTS:
 *
 * OUTPUTS;
 *		If successful, returns a handle to the Sound Events key. It is up
 *		to the caller to subsequently close this.
 *
 *		If unsuccessful, returns 0.
 */

TCHAR		SoundSchemeKeyName[] = "AppEvents\\Schemes\\Apps";

HKEY openSoundSchemeKey()
{
	HKEY	schemeKey;

	/* Open the HKEY_CURRENT_USER\\AppEvents\\Schemes\\Apps key */
	if (!RegOpenKeyEx(HKEY_CURRENT_USER, &SoundSchemeKeyName[0], 0, KEY_ALL_ACCESS, &schemeKey))
	{
		return(schemeKey);
	}

	return(0);
}





/**************************** createAppKey() ****************************/
/* This creates the App key with the desired keyname and label,
 *
 * INPUTS:
 *		keyname - A pointer to the null-terminated name of the App key
 *		label - A pointer to the null-terminated label displayed to the user
 *				when he uses the Control Panel's Sound utility.
 *		soundSchemeKey -- A pointer to a HKEY variable which will receive the open
 *				handle of the HKEY_CURRENT_USER\AppEvents\Schemes\Apps key.
 *				You'll need this to pass to deleteKey() in case of an error.
 *
 * OUTPUTS;
 *		If successful, returns a handle to the created App key, and
 *		phKey will be set to the handle of the
 *		HKEY_CURRENT_USER\AppEvents\Schemes\Apps key.
 *
 *		If unsuccessful, returns 0.
 */

HKEY createAppKey(LPTSTR keyname, LPTSTR label, PHKEY soundSchemeKey)
{
	HKEY		appKey;
	DWORD		len;
	
	/* Open the HKEY_CURRENT_USER\\AppEvents\\Schemes\\Apps key */
	if ((*soundSchemeKey = openSoundSchemeKey()))
	{
		/* Create the application key */
		if (!RegCreateKeyEx(*soundSchemeKey, keyname, 0, keyname,
			REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &appKey, &len))
		{
			/* Set the app key's default value to the string I want
			   displayed in the Control Panel's Sound utility as the application heading */
			if (!RegSetValueEx(appKey, 0, 0, REG_SZ, label,lstrlen(label) + sizeof(TCHAR)))
			{
				return(appKey);
			}

			/* An error, so close App key handle */
			RegCloseKey(appKey);

			/* Delete App key */
			RegDeleteKey(soundSchemeKey, keyname);
		}

		/* An error, so close HKEY_CURRENT_USER\\AppEvents\\Schemes\\Apps key handle */
		RegCloseKey(*soundSchemeKey);
	}

	return(0);
}





/*************************** createSoundAction() **************************/
/* This creates a Sound Action subkey with the desired keyname and label,
 * and sets its .current subkey's default value to the name  of the
 * desired WAVE file.
 *
 * INPUTS:
 *		appKey - The handle of the app key (under which this Sound Action
 *				subkey is created).
 *		keyname - A pointer to the null-terminated name of the Sound Action
 *				subkey.
 *		label - A pointer to the null-terminated label displayed to the user
 *				when he uses the Control Panel's Sound utility.
 *		wave - A pointer to the null-terminated name of the desired WAVE file
 *				associated with this Sound Action.
 *
 * OUTPUTS;
 *		Returns a 0 for success, non-zero for an error. In case of an error,
 *		the caller must do its own registry cleanup.
 */

TCHAR		Current[] = ".current";

DWORD createSoundAction(HKEY appKey, LPTSTR keyname, LPTSTR label, LPTSTR wave)
{
	HKEY		subKey, currentKey;
	DWORD		len;

	/* Create the Sound Action subkey */
	if (!(len = RegCreateKeyEx(appKey, keyname, 0, keyname,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &subKey, &len)))
	{
		/* Set subkey's default value to the label I want displayed
		   in Control Panel's Sound utility for this Sound Action */
		if (!(len = RegSetValueEx(subKey, 0, 0, REG_SZ, label, lstrlen(label) + sizeof(TCHAR))))
		{
			/* Create the .current subkey under Sound Action */
			if (!(len = RegCreateKeyEx(subKey, &Current[0], 0, &Current[0],
				REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &currentKey, &len)))
			{
				/* Set .current subkey's default value to the name of the desired WAVE file */
				len = RegSetValueEx(currentKey, 0, 0, REG_SZ, wave, lstrlen(wave) + sizeof(TCHAR));

				/* Close .current subkey */
				RegCloseKey(currentKey);
			}
		}

		/* Close Sound Action subkey */
		RegCloseKey(subKey);
	}

	return(len);
}






/***************************** keyNameCopy() ****************************/
/* Copies a key name to a buffer, appends '\' char, and returns the end of
 * that buffer.
 *
 * INPUTS:
 *		toStr - A pointer to the buffer where the key name is copied.
 *		fromStr - A pointer to the null-terminated key name to be copied.
 *
 * OUTPUTS;
 *		Returns the end of the buffer.
 */

PTCHAR keyNameCopy(PTCHAR toStr, LPTSTR fromStr)
{
	while ((*(toStr)++ = *(fromStr)++));
	*(toStr - 1) = '\\';
	return(toStr);
}





/*************************** playSoundAction() **************************/
/* This plays a Sound Action.
 *
 * INPUTS:
 *		keyname - A pointer to the null-terminated name of the Sound Action
 *				subkey.
 *		flags - The flags that normally get passed to PlaySound(). Here you
 *				would likely pass in either 0 or OR'ed flags such as SND_ASYNC,
 *				SND_LOOP, SND_NODEFAULT, SND_NOSTOP, or SND_NOWAIT. Do not
 *				pass in other flags!
 *
 * OUTPUTS;
 *		Returns a 1 for success, 0 for an error.
 */

DWORD playSoundAction(LPTSTR keyname, DWORD flags)
{
	TCHAR	waveName[MAX_PATH];
	LPTSTR	ptr;
	DWORD	len, type;
	HKEY	hKey;

	/* Construct the name of the Sound Action registry key */
	ptr = keyNameCopy(&waveName[0], &SoundSchemeKeyName[0]);
	ptr = keyNameCopy(ptr, &AppKeyName[0]);
	ptr = keyNameCopy(ptr, keyname);
	ptr = keyNameCopy(ptr, &Current[0]);
	*(ptr - 1) = 0;

	/* Open its .current key */
	if (!RegOpenKeyEx(HKEY_CURRENT_USER, &waveName[0], 0, KEY_ALL_ACCESS, &hKey))
	{
		/* Get the default value of its .current key (ie, the wave file name) */
		len = MAX_PATH;
		type = REG_SZ;
		if (!RegQueryValueEx(hKey, 0, 0, (LPDWORD)&type, (LPBYTE)&waveName[0], &len))
		{
			/* Close .current subkey */
			RegCloseKey(hKey);
			
			/* Play the WAVE, returning PlaySound()'s error code */
			return(PlaySound(&waveName[0], 0, flags));
		}

		/* Close .current subkey */
		RegCloseKey(hKey);
	}

	/* An error */
	return(0);
}





/********************************** main() *******************************/

int main(int argc, char **argv)
{
	HKEY		soundSchemeKey, appKey;

	/* =========================================================== */
	/* ======== Create/Setup my program's Sound Action keys ======= */
	/* =========================================================== */

	/*	This setup of Sound Action keys would normally be done only when
		your program is first installed. The idea is to setup these registry
		keys once when your program is first installed, and then subsequently
		use them whenever your app does a PlaySound(). The user will be able
		to modify them later using Control Panel's Sound utility.

		Here I just add/delete them every time the program runs. That defeats the
		whole purpose of allowing the user to change them, but this is just an
		example to show you how to setup, use, and delete these keys.
	*/

	/* Create the app key */
	if ((appKey = createAppKey(&AppKeyName[0], &AppStrName[0], &soundSchemeKey)))
	{
		/* Create/Setup the ProgramStart Sound Action subkey */
		if (createSoundAction(appKey, &ProgramStart[0], &ProgramStartStr[0], &WaveName[0]))
		{
bad:		printf("Couldn't create Sound Action registry subkey!\r\n");

			/* Close App subkey */
			RegCloseKey(appKey);
				
			/* Delete registry keys */
			deleteKey(soundSchemeKey, &AppKeyName[0]);
				
			/* Close Scheme key */
			RegCloseKey(soundSchemeKey);

			/* Exit the program */
			return(2);
		}

		/* Create/Setup the ProgramEnd Sound Action subkey */
		if (createSoundAction(appKey, &ProgramEnd[0], &ProgramEndStr[0], &WaveName[0])) goto bad;

		/* Create/Setup the SoundEffect1 Sound Action subkey */
		if (createSoundAction(appKey, &SoundEffect1[0], &SoundEffect1Str[0], &WaveName[0])) goto bad;

		/* Close App key */
		RegCloseKey(appKey);

		/* Close Scheme key */
		RegCloseKey(soundSchemeKey);
	}
	else
	{
		printf("Couldn't create the Application registry key!\r\n");

		/* Exit the program */
		return(1);
	}






	/* =========================================================== */
	/* ============ Use my program's Sound Action keys =========== */
	/* =========================================================== */

	/*	Here is what a program would normally do -- simply use its
		Sound Actions with PlaySound()
	*/

	/* Play the ProgramStart Sound Action and wait for it to finish */
	if (!playSoundAction("ProgramStart", 0))
	{
		printf("The ProgramStart WAVE didn't play!\r\n");
	}

	/* Now play the SoundEffect1 everytime the user presses any key except the SPACE BAR */

	/*	As an experiment here, while this program is waiting for input, go over to Control Panel's
		Sound utility, and try changing around the WAVE file associated with the "Sound Effect 1"
		under the "PlaySound Example" heading. Then come back to this program's window and see
		how that translates to a different WAVE file being used. See? You can also change the
		"When program ends" WAVE file, to hear the change when this program exits.
	*/

	for (;;)
	{
		int		chr;

		chr = getch();

		if (chr == 0x20) break;	/* Exit when SPACE BAR pressed */

		if (!playSoundAction("SoundEffect1", SND_ASYNC))
		{
			printf("The Sound Effect WAVE didn't play!\r\n");
		}
	}

	/* Play the ProgramEnd Sound Action and wait for it to finish */
	if (!playSoundAction("ProgramEnd", 0))
	{
		printf("The ProgramEnd WAVE didn't play!\r\n");
	}






	/* =========================================================== */
	/* ========== Delete my program's Sound Action keys ========== */
	/* =========================================================== */

	/*	This deletion of Sound Action keys would normally be done only when
		your program is uninstalled. I do it here because this example
		"installs" itself whenever it runs, by creating/setting up those
		Sound Action registry keys. Remember, normally that would be
		done once only when the program is first installed.
	*/

	/* Open the HKEY_CURRENT_USER\\AppEvents\\Schemes\\Apps key */
	if ((soundSchemeKey = openSoundSchemeKey()))
	{
		/* Delete registry keys */
		deleteKey(soundSchemeKey, &AppKeyName[0]);
				
		/* Close Scheme key */
		RegCloseKey(soundSchemeKey);
	}

	return(0);
}
