/* MidSkele -- A minimal Windows C MIDI program
 *
 * Handles the selection of the MIDI Input/Output devices. Functions such as OpenMidiIn()/CloseMidiIn()/etc
 * are called in our midi_io.dll Dynamic Link Library to open/close MIDI devices.
 */

/* system includes */
#include <windows.h>
#include <windowsx.h>

/* app includes */
#include "resource.h"
#include "globals.h"
#include "midi_io.h"






/* ************* Globals ************** */

/* Strings for MIDI input */
const char		MidiInTitle[] = "Choose Midi Input:";

/* Strings for MIDI output */
const char		MidiOutTitle[] = "Choose Midi Output:";







/***************************** midiInputEvt() *******************************
 * Here's my MIDI Input callback that Windows calls whenever 1 of 4 possible 
 * things happen:
 *
 * 1).	I open a MIDI In Device via midiInOpen(). In this case, the
 *		uMsg arg to my callback will be MIM_OPEN. The handle arg will
 *		be the same as what is returned from midiInOpen(). The
 *		dwInstance arg is whatever I passed to midiInOpen() as its
 *		dwInstance arg.
 *
 * 2).	I close a MIDI In Device via midiInClose(). In this case, the
 *		uMsg arg to my callback will be MIM_CLOSE. The handle arg will
 *		be the same as what was passed to midiInClose(). The
 *		dwInstance arg is whatever I passed to midiInClose() as its
 *		dwInstance arg when I initially opened this handle.
 *
 * 3).	One, regular (ie, everything except System Exclusive messages) MIDI
 *		message has been completely input. In this case, the uMsg arg to my
 *		callback will be MIM_DATA. The handle arg will be the same as what
 *		is passed to midiInOpen(). The dwInstance arg is whatever I passed
 *		to midiInOpen() as its dwInstance arg when I initially opened this
 *		handle. The dwParam1 arg is the bytes of the MIDI Message packed
 *		into an unsigned long in the same format that is used by
 *		midiOutShort(). The dwParam2 arg is a time stamp that the device
 *		driver created when it recorded the MIDI message.
 *
 * 4).	midiInOpen has either completely filled a MIDIHDR's memory buffer
 *		with part of a System Exclusive message (in which case we had better
 *		continue queuing the MIDIHDR again in order to grab the remainder
 *		of the System Exclusive), or the MIDIHDR's memory buffer contains the
 *		remainder of a System Exclusive message (or the whole message if it
 *		happened to fit into the memory buffer intact). In this case, the
 *		uMsg arg to my callback will be MIM_LONGDATA. The handle arg will be
 *		the same as what is passed to midiInOpen(). The dwInstance arg is
 *		whatever I passed to midiInOpen() as its dwInstance arg when I
 *		initially opened this handle. The dwParam1 arg is a pointer to the
 *		MIDIHDR whose memory buffer contains the System Exclusive data. The
 *		dwParam2 arg is a time stamp that the device driver created when it
 *		recorded the MIDI message.
 *
 * 5).	This callback is not processing data fast enough such that the MIDI
 *		driver (and possibly the MIDI In port itself) has had to throw away
 *		some incoming, regular MIDI messages. In this case, the uMsg arg to my
 *		callback will be MIM_MOREDATA. The handle arg will be the same as what
 *		is passed to midiInOpen(). The dwInstance arg is whatever I passed
 *		to midiInOpen() as its dwInstance arg when I initially opened this
 *		handle. The dwParam1 arg is the bytes of the MIDI Message that was
 *		not handled (by an MIM_DATA call) packed into an unsigned long in the
 *		same format that is used by midiOutShort(). The dwParam2 arg is a time
 *		stamp that the device driver created when it recorded the MIDI message.
 *		In handling a series of these events, you should store the MIDI data
 *		in a global buffer, until such time as you receive another MIM_DATA
 *		(which indicates that you can now do the more time-consuming processing
 *		that you obviously were doing in handling MIM_DATA).
 *		NOTE: Windows sends an MIM_MOREDATA event only if you specify the
 *		MIDI_IO_STATUS flag to midiInOpen().
 *
 * 6).	An invalid, regular MIDI message was received. In this case, the uMsg
 *		arg to my callback will be MIM_ERROR. The handle arg will be the same
 *		as what is passed to midiInOpen(). The dwInstance arg is whatever I
 *		passed to midiInOpen() as its dwInstance arg when I initially opened
 *		this handle. The dwParam1 arg is the bytes of the MIDI Message that was
 *		not handled (by an MIM_DATA call) packed into an unsigned long in the
 *		same format that is used by midiOutShort(). The dwParam2 arg is a time
 *		stamp that the device driver created when it recorded the MIDI message.
 *
 * 7).	An invalid, System Exclusive message was received. In this case, the uMsg
 *		arg to my callback will be MIM_LONGERROR. The handle arg will be the same
 *		as what is passed to midiInOpen(). The dwInstance arg is whatever I
 *		passed to midiInOpen() as its dwInstance arg when I initially opened
 *		this handle. The dwParam1 arg is a pointer to the MIDIHDR whose memory
 *		buffer contains the System Exclusive data. The dwParam2 arg is a time
 *		stamp that the device driver created when it recorded the MIDI message.
 *
 * The time stamp is expressed in terms of milliseconds since your app
 * called midiInStart().
 *
 * In this particular function, I simply do nothing with the MIDI input. It's
 * merely a skeleton callback.
 *************************************************************************/

void CALLBACK midiInputEvt(HMIDIIN handle, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
	if (MidiInHandle)
	{
		/* Determine why Windows called me */
		switch (uMsg)
		{
			/* Received some regular MIDI message */
			case MIM_DATA:
			{
				return;
			}

			/* Received all or part of some System Exclusive message */
			case MIM_LONGDATA:
			{
				return;
			}
		}
	}
}





/************************ enumMidiOutDevs() **************************
 * Fills in the MIDI Out device listbox with the names of all MIDI Out
 * devices in the computer.
********************************************************************/

void enumMidiOutDevs(HWND hwndChild)
{
	MIDIOUTCAPS		moc;
	unsigned long	iNumDevs;

	/* Get the number of MIDI In devices in this computer */
	iNumDevs = midiOutGetNumDevs();

	/* Go through all of those devices, adding their names to the MIDI In devices listbox */
	while (iNumDevs--)
	{
		/* Get info about the next device */
		if (!midiOutGetDevCaps(iNumDevs, &moc, sizeof(MIDIOUTCAPS)))
		{
			/* Add its name to the listbox */
			SendMessage(hwndChild, LB_INSERTSTRING, 0, (LPARAM)&moc.szPname);

			/* Select this name in the listbox if its the currently selected Output */
			if (iNumDevs == MidiOutID) SendMessage(hwndChild, LB_SELECTSTRING, iNumDevs-1, (LPARAM)&moc.szPname);
		}
	}
}





/************************ enumMidiInDevs() **************************
 * Fills in the MIDI In device listbox with the names of all MIDI In
 * devices in the computer.
 ********************************************************************/

void enumMidiInDevs(HWND hwndChild)
{
	MIDIINCAPS		mic;
	unsigned long	iNumDevs;

	/* Get the number of MIDI In devices in this computer */
	iNumDevs = midiInGetNumDevs();

	/* Go through all of those devices, adding their names to the MIDI In devices listbox */
	while (iNumDevs--)
	{
		/* Get info about the next device */
		if (!midiInGetDevCaps(iNumDevs, &mic, sizeof(MIDIINCAPS)))
		{
			/* Add its name to the listbox */
			SendMessage(hwndChild, LB_INSERTSTRING, 0, (LPARAM)&mic.szPname);

			/* Select this name in the listbox */
			if (iNumDevs == MidiInID) SendMessage(hwndChild, LB_SELECTSTRING, iNumDevs-1, (LPARAM)&mic.szPname);
		}
	}
}





/***************************** midiDevDlgProc() **************************
 * The message handler for the MIDI Output or Input Device Dialog box.
 *************************************************************************/

BOOL CALLBACK midiDevDlgProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	HWND	hwndChild;

	switch(uMsg)
	{
		case WM_COMMAND:
		{
			/* There is only the one listbox, so if we get a WM_COMMAND, it must be
			 * the listbox. Hence, we'll skip checking the LOWORD of wParam for the
			 * ID of the listbox.
			 */

			/* Doubleclick in the listbox? If not, ignore this */
			if(HIWORD(wParam) != LBN_DBLCLK) return(0);

			/* Get the MIDI Out devices listbox */
			if ((hwndChild = GetDlgItem(hwndDlg, IDC_MIDILIST)))
			{
				/* Get the select Device ID */
				if (LB_ERR != (uMsg = SendMessage(hwndChild, LB_GETCURSEL, 0, 0)))
				{
					/* Input or Output list? */
					if (GetWindowLong(hwndDlg, DWL_USER))
					{
						/* Close any previously opened MIDI output device */
						CloseMidiOut(MainWindow);

						/* Save Device ID in global */
						MidiOutID = uMsg;

						/* Open the new MIDI output */
						OpenMidiOut(MainWindow);
					}

					else
					{
						/* Close any previously opened MIDI input device */
						CloseMidiIn(MainWindow);

						/* Save Device ID in global */
						MidiInID = uMsg;

						/* Open the new MIDI input */
						OpenMidiIn(MainWindow, (DWORD)midiInputEvt);
					}
				}
			}
		}

		/* Fall through to close the Dialog */

		/* ================== User wants to close window ==================== */
		case WM_CLOSE:
		{
			/* Close the Dialog */
			EndDialog(hwndDlg, 0);

			break;
		}

		/* ======================= Dialog Initialization ===================== */
		case WM_INITDIALOG:
		{
			/* Save the Input/Output flag in the dialog's DWL_USER field */
			SetWindowLong(hwndDlg, DWL_USER, lParam);

			/* Get the MIDI listbox */
			if ((hwndChild = GetDlgItem(hwndDlg, IDC_MIDILIST)))
			{
				/* Input or Output list? */
				if (lParam)
				{
					/* Fill in the listbox with Output Devices and highlight current selection */
					enumMidiOutDevs(hwndChild);

					/* Set dialog title */
					SetWindowText(hwndDlg, &MidiOutTitle[0]); 
				}

				else
				{
					/* Fill in the listbox with Input Devices and highlight current selection */
					enumMidiInDevs(hwndChild);

					/* Set dialog title */
					SetWindowText(hwndDlg, &MidiInTitle[0]); 
				}
			}

			/* Let Windows set control focus */
			return(1);
		}
	}

	return(0);
}




/****************************** doMidiDevDlg() ****************************
 * Opens and operates the MIDI Output or Input Device Dialog box.
 *
 * type =	1 for Output MIDI device, or 0 for Input MIDI device.
 **************************************************************************/

void doMidiDevDlg(LPARAM type)
{
	/* Create and operate the Dialog */
	if (DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_MIDILIST), MainWindow, midiDevDlgProc, type))
	{
		MessageBeep(0xFFFFFFFF);
	}
}





/**************************** doMidiOutDevDlg() **************************
 * Opens and operates the MIDI Out Device Dialog box.
 *************************************************************************/

void doMidiOutDevDlg(void)
{
	/* Create and operate the Dialog */
	doMidiDevDlg(1);
}





/**************************** doMidiInDevDlg() **************************
 * Opens and operates the MIDI In Device Dialog box.
 ************************************************************************/

void doMidiInDevDlg(void)
{
	/* Create and operate the Dialog */
	doMidiDevDlg(0);
}
