/* Example DLL called by MIDIPGMS.EXE to Import or Export to the GENMIDI database */

/* This must be linked with genmidi.lib */

/* OS includes */
#include <windows.h>
#include <string.h>
#include <stdio.h>
#include "..\genmidi.h"
#include "resource.h"





/********************************* DllMain() *******************************
 * Automatically called when the DLL is loaded. We don't need to do anything
 * much here, so we simply return 1. Remember that this function's name
 * must be supplied on the linker link with the -entry flag.
 ***************************************************************************/

BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	/* Success */
	return(1);
}





/****************************** trim_head() ******************************
 * Skips leading whitespace in passed null-terminated string and returns
 * ptr to first non-space char.
 *************************************************************************/

unsigned char * trim_head(unsigned char * str)
{
    while (isspace(*str)) str++;
	return(str);
}





/***************************** trim_spaces() *****************************
 * Trims leading and trailing whitespace from passed null-terminated string.
 * Also trims ending newline. Returns end of line.
 *************************************************************************/

char * trim_spaces(unsigned char * str)
{
	unsigned char *	ptr;
	unsigned char *	end = str;
	unsigned char	chr;

	/* Skip leading spaces */
	ptr = trim_head(str);

	do
	{
		/* Move next char up to head of string */
		chr = *(ptr)++;
		*(str)++ = chr;

		/* If not a space or newline, advance the end of string pointer */
		if (chr && (chr != 10) && !isspace(chr))
		{
			end = str;
		}

	} while (chr);

	/* Null-terminate */
	*end = 0;

	return(end);
}





/******************************* MidiImport() ******************************
 * Implements importing to the current GenMidi database.
 *
 * Format of the source text file that this app converts:
 *
 * Blank lines, or lines that begin with a ; character are ignored. (ie, To
 * place comments in the text file, begin the line with a ; character). Leading
 * and trailing spaces are ignored on all names. Each name must appear upon one
 * line. (ie, Don't let a bank name span several lines).
 *
 * An Instrument name is placed between the curly braces { and }. For example:
 *
 * { Roland JV-1080 }
 *
 * After an instrument name is the name of its first bank, followed by the patch
 * names in that bank. Then, the second bank for that instrument follows. After
 * all of the banks (and their patches) for an instrument are listed, the next
 * instrument follows.
 *
 * A bank's number, followed by its name, are placed between the braces [ and ].
 * For example (a bank named "First Bank" whose bank number, sent over MIDI using
 * Bank Select Controller, is 0):
 *
 * [ 0 First Bank ]
 *
 * Optionally, the bank number can be omitted. In this case, the bank number is
 * simply incremented from the previous bank's number (with the first bank being
 * 0). ie, The first listed bank is 0, the second is 1, the third is 2, etc).
 *
 * If an instrument uses only one controller (ie, #0 or #32, but not both) to
 * change banks, then simply list the values for each controller separately, and
 * set the one not used to -1. The value for controller 0 appears first, and the
 * value for controller 32 appears second. For example, here we indicate that, for
 * this bank, the value for controller 0 is 10, and controller 32 isn't used at all
 * when switching banks:
 *
 * [10 -1 First Bank]
 *
 * If a bank number is specified in hexadecimal, use C notation (ie, prepend 0x
 * characters to the value).
 *
 * To specify a dummy bank number (ie, neither controller are used, then simply
 * use -1 alone. For example:
 *
 * [-1 First Bank]
 *
 * Simply list a patch's MIDI Program number, followed by its name. For example (a
 * patch named "Third Patch" whose program number is 2):
 *
 * 2 Third Patch
 *
 * Optionally, the program number can be omitted. In this case, the number is simply
 * incremented from the previous patch's number (with the first patch being 0). ie,
 * The first listed patch is 0, the second is 1, the third is 2, etc). Also an = sign
 * may optionally be placed inbetween the number and name. This allows easily
 * importing Patch Names from CakeWalk's .INS text files.
 *
 * Caveats: Bank names can't begin with a numeric character if the bank number is
 * omitted. Patch names can't begin with a numeric character (if the patch number is
 * omitted), = sign (unless the patch number is omitted), ; character, nor { or [
 * braces.
 ***************************************************************************/

__declspec(dllexport) long MidiImport(char * Filename)
{
	FILE *			source;
	char *			ptr;
	char *			end;
	long			result;
	unsigned short	bankNum;
	unsigned char	insNum, pgmNum;

	/* Open the text file */
	if ((source = fopen(Filename, "rt")))
	{
		/* Not handling an Instrument yet */
		insNum = 0xFF;

		/* Lock database */
		if (!MidiLock(MIDILOCK_MODIFY, MIDILOCK_WAITFOREVER))
		{
			/* Get next line of text file. Use Filename buffer to read in the line */
			while ((ptr = fgets(Filename, MAX_PATH, source)))
			{
				/* Remove leading spaces, trailing spaces, and final newline */
				end = trim_spaces(ptr) - 1;

				/* Is this an Instrument name? (ie, begins with '{' char) */
				if (*ptr == '{')
				{
					/* Skip { char */
					++ptr;

					/* Trim } char */
					if (*end != '}')
					{
						/* "No '}' after Instrument name!" */
						result = IDS_NOENDINS;
						goto badimp;
					}
					*end = 0;

					/* Add an Instrument with this name. Use default flags */
					if ((result = MidiAddInstrument(0, ptr, MIDI_PRESSRESET|MIDI_PITCHRESET)))
					{
						/* Close Text file */
badimp:					fclose(source);
		
						/* Unlock database */
						MidiUnlock();

						/* Return the error that GENMIDI.DLL returned */
						return(result);
					}

					/* Get the Instrument's number */
					insNum = (unsigned char)MidiGetInstrumentNum(ptr);

					/* Haven't got a Bank yet */
					bankNum = (unsigned short)-1;
				}

				/* For Bank or Program names, make sure that we have an Instrument first */
				else if (insNum != 0xFF)
				{
					/* Is this a Bank name? (ie, begins with '[' char) */
					if (*ptr == '[')
					{
						/* Skip [ char */
						++ptr;

						/* Trim ] char */
						if (*end != ']')
						{
							/* "No ']' after Bank name!" */
							result = IDS_NOENDBANK;
							goto badimp;
						}
						*end = 0;

						/* Bank number specified? */
						ptr = trim_head(ptr);
						if (isdigit(*ptr))
						{
							/* Get the bank number */
							bankNum = (unsigned short)strtoul(ptr, &ptr, 0);
							ptr = trim_head(ptr);

							/* LSB number also specified? (ie, controller #32) */
							if (!isdigit(*ptr)) goto incbank;

							/* Store the previous number as MSB (ie, controller #0) */
							if (bankNum & 0x0080) bankNum = 0;
							bankNum <<= 7;

							pgmNum = (unsigned char)strtoul(ptr, &ptr, 0);
							if (pgmNum < 0x80) bankNum |= pgmNum;
							ptr = trim_head(ptr);
						}
						else
						{
							/* Increment previous bankNum. Could even be a dummy Bank number */
							bankNum++;
						}

incbank:				/* Create a new bank */						
						if ((result = MidiAddBank(MidiNames.Instrument, 0, ptr, MidiNumToBank((short)bankNum), 0))) goto badimp;

						/* Reinit Program number for first Program */
						pgmNum = (unsigned char)-1;
					}

					/* Must be Program name */
					else if (bankNum != 0xFFFF)
					{
						/* If a blank line, or comment (ie, begins with ; char), or we're not
						   writing a bank, skip this line */
						if (!(*ptr) || (*ptr == ';')) continue;

						/* Is the program number specified? */
						if (isdigit(*ptr))
						{
							/* Get program number */
							pgmNum = (unsigned char)strtoul(ptr, &ptr, 0);
							ptr = trim_head(ptr);

							/* Skip any '=' sign after program number (so CakeWalk .INS files can be
							   imported) */
							if (*ptr == '=') ptr++;
						}
						else
						{
							/* If no program number, then just increment num */
							pgmNum++;
						}

						/* "This Program has a number > 127!" */
						if (pgmNum > 127)
						{
							result = IDS_PGMTOOHIGH;
							goto badimp;
						}

						/* Add a Program */
						if ((result = MidiAddPgm(MidiNames.Instrument, MidiNames.Bank, ptr, pgmNum, 0))) goto badimp;
					}
				}
			}

			/* Unlock database */
			MidiUnlock();

			/* Success */
			return(0);
		}
	}

	return(IDS_NOIMPORT);
}




/******************************* MidiExport() ******************************
 * Implements exporting to the current GenMidi database. Actually, this does
 * nothing, so we should really eliminate it from the DLL.
 ***************************************************************************/

__declspec(dllexport) long MidiExport(char * Filename)
{
	/* Success */
	return(0);
}





/****************************** MidiImExErr() ******************************
 * Returns an error message for one of our error numbers. This DLL should
 * not generate any error numbers less than 1000, although it may return
 * such numbers from calls to GENMIDI.DLL functions.
 ***************************************************************************/

__declspec(dllexport) void MidiImExErr(char * Buffer, unsigned long BufferLength, long ErrorNum)
{

	/* Null-terminate buffer */
	*Buffer = 0;
}





/***************************** MidiExtension() *****************************
 * This function should exist only if you want MIDIPGMS.EXE to present a
 * File Dialog, get the name of the file to import or export, and pass that
 * to MidiImport or MidiExport. Otherwise, omit this function if your
 * MidiImport and MidiExport functions get their own filename.
 *
 * This function should copy the desired File Dialog Extension into the
 * passed Extension buffer (with a length of MAX_PATH), and then return the
 * length of the null-terminated string (not counting the null byte). For
 * example, to specify giving the user the option to display files that end
 * in .TXT or all files, the string would be:
 *
 * "Text files (*.txt)|*.txt|All files (*.*)|*.*||"
 *
 * If you want MIDIPGMS to display the File Dialog and get the filename, but
 * you want all files displayed, then simply return a 1 here.
 *
 * If you have a error returning the Extension string, return 0.
 ***************************************************************************/

const char DllExt[] = "Text files (*.txt)|*.txt|All files (*.*)|*.*||";

__declspec(dllexport) unsigned long MidiExtension(char * Extension)
{
	strcpy(Extension, &DllExt[0]);

	return(sizeof(DllExt) - 1);
}
