/* PgmName.c
 *
 * An example of using GENMIDI.DLL.
 *
 * This example shows how to call the DLL function MidiGetPgmNum() to retrieve
 * the MIDI program number for a given Program name. When running the program,
 * supply a defined Program Name. The program will print the respective MIDI
 * Program Change number for that name if a match is found. Otherwise,
 * a message is printed that the name is not found.
 *
 * By supplying an Instrument and Bank name, this also shows how to use
 * MidiGetInstrumentNum(), MidiNextInstrument(), MidiNextBank(), and
 * MidiGetBankNum().
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "..\genmidi.h"


/* ********************************* main() ********************************* */

void main(int argc, char *argv[])
{
	unsigned char * insPtr;
	unsigned char * bankPtr;
	unsigned char	numInstruments;
	unsigned char	insNum;
	unsigned short	bankNum;
	unsigned char	pgm;

	/* Check if any parameters were given, if not, display some help info */
	if (argc < 2)
	{
		_cputs("Displays the program number for a specified Program Name.\r\n\nSyntax is '");
		_cputs(argv[0]);
		_cputs( " Program Instrument Bank'.\r\n\twhere Program is a program name (use double quotes if it contains spaces).\r\n\t");
		_cputs( "For example, \"Grand Piano\".\r\n\t");
		_cputs( "where Instrument is an instrument name (use double quotes if it contains spaces).\r\n\t");
		_cputs( "If no instrument name is specified, then all instruments are searched.\r\n\t");
		_cputs( "where Bank is a bank name (use double quotes if it contains spaces).\r\n\t");
		_cputs( "If no bank name is specified, then all banks are searched in the specified instrument.\r\n\n");
		_cputs("Version 1.0\r\n");
		exit(0);
	}

	/* Assume all 255 possible instruments will be searched */
	numInstruments = 255;
	bankNum = 0;
	insNum = 0;

	/* Indicate that we're accessing the database, but don't lock out other programs */
	if (!MidiLock(MIDILOCK_ACCESS, MIDILOCK_WAITFOREVER))
	{
		/* Specified an instrument? */
		if (argc > 2)
		{
			/* Get instrument number based upon specified name */
			insNum = (unsigned char)MidiGetInstrumentNum(argv[2]);

			/* Found it? */
			if (insNum == 255)
			{
				MidiUnlock();
				_cputs("ERROR: Can't find '");
				_cputs(argv[2]);
				_cputs("' Instrument in DLL database!\r\n");
				exit(-1);
			}

			/* Only one instrument to be searched */
			numInstruments = 1;

			/* Get the Instrument's name in the DLL database. This was set by the DLL
			in the above call to MidiGetInstrumentNum() */
			insPtr = MidiNames.Instrument;

			/* Specified a bank too? */
			if (argc > 3)
			{
				/* Get bank number based upon specified name */
				bankNum = (unsigned short)MidiGetBankNum(argv[3], insNum);

				/* Found it? */
				if (bankNum == 0xFFFF)
				{
					MidiUnlock();
					_cputs("ERROR: Can't find '");
					_cputs(argv[3]);
					_cputs("' Bank in DLL database!\r\n");
					exit(-2);
				}
			}
		}
		else
		{
			/* Get the first Instrument's name and number */
			insPtr = 0;
			if ((insNum = (unsigned char)MidiNextInstrument(&insPtr)) == 0xFF)
			{
				MidiUnlock();
				_cputs("ERROR: No Instruments in DLL database!");
				exit(-3);
			}
		}

		/* Search specified instruments */
		do
		{
			bankPtr = 0;

			/* Search specified banks */
			do
			{
				/* If user didn't specify a bank #, then we need to get the next bank's # */
				if (argc < 3)
				{
					/* Get the next bank's FLAGS/BANKNUM */
					bankNum = (unsigned short)MidiNextBank(&bankPtr, insPtr);

					/* No more banks? */
					if (bankNum == 0xFFFF) break;
				}

				/* Find a match (within this bank) to the supplied Program Name */
				pgm = MidiGetPgmNum(argv[1], bankNum, insNum);

				/* Is this a defined Program name? If so, then the returned program number isn't 0xFF */
				if (pgm != 0xFF)
				{
					/* Print the program number (and also use the full Program name from the DLL database
					in case the user supplied a substring). Also, print the full Instrument and Bank names
					set by the DLL in the following call to MidiGetPgmName(). */

					/* Print the instrument name */
					_cputs("\r\n===============================================================\r\n");
					_cputs(MidiNames.Instrument);
					_cputs("\r\n");

					/* Print the Bank name */
					_cputs("\t-------------------------------------------------------\r\n\t");
					_cputs(MidiNames.Bank);
					_cputs("\r\n\t-------------------------------------------------------\r\n");

					/* Print program number and name */
					printf("\"%s\" is program %u. (%u if 0 is considered the first patch).\r\n", MidiNames.Pgm, pgm+1, pgm);
				}

			/* Search another bank? */
			} while (argc < 3);

		/* Search another instrument? Is there another instrument? */
		} while ((--numInstruments) && ((insNum = (unsigned char)MidiNextInstrument(&insPtr)) != 0xFF));

		/* Indicate that we're done accessing the database */
		MidiUnlock();
	}

	exit(0);
}
