/* LoadPgms.c
 *
 * An example of using GENMIDI.DLL.
 *
 * This example shows how to call the DLL function MidiLoadDat()
 * to load an Instrument/Bank/Program to the DLL's database.
 *
 * It also demonstrates how to traverse the entire DLL database, listing
 * each instrument with its banks, programs, and note names.
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "..\genmidi.h"

long	lines = 0;
long	count = 1;


/************************** check_pause() ****************************
 * Check if we have filled the screen. If so, pause output waiting for
 * the user to press a key to continue.
 *********************************************************************/

void check_pause(long more)
{
	/* User wants the pause feature (ie, lines is not 0), and we filled the screen? */
	count += more;
	if (lines && (count >= lines))
	{
		count = 1;
		_cputs("--- Press ENTER to continue ---");
		getchar();
		_cputs("\r\n");
	}
}



/***************************** main() ********************************/

void main(int argc, char *argv[])
{
	unsigned char *	insName;
	unsigned char *	bankName;
	unsigned char *	pgmName;
	unsigned char *	noteName;
	unsigned char *	pgmStruct;
	unsigned char *	noteStruct;
	unsigned long	loop;
	unsigned char	buffer[120];

	/* Check if any parameters were given, if not, display some help info */
	if (argc < 2)
	{
		_cputs("Loads an Instrument/Bank/Program.\r\n\nSyntax is '");
		_cputs("'LOADPGMS Filename [Operation]'.\r\n\t where Filename is the file to load.\r\n");
		_cputs("\t where Operation is 1 for an instrument, 2 for a bank, or 3 for a program.\r\n");
		_cputs("If Operation is not supplied, then an Instrument is assumed.\r\n");
		_cputs("Use double quotes if a Filename contains spaces.\r\n");
		_cputs("Version 1.0\r\n");
		exit(0);
	}

	/* Parse command line args */
	buffer[0] = 0;
	for (loop=2; loop < (unsigned long)argc; loop++)
	{
		/* An option? */
		bankName = argv[loop];
		if (*bankName == '/')
		{
			bankName++;
			switch ((*bankName) & 0x5F)
			{
				/* Screen lines */
				case 'L':
					lines = atol(bankName+1);
			}
		}

		/* Operation */
		else
		{
			buffer[0] = (unsigned char)atoi(bankName);
		}
	}

	/* Lock the database before changes. Indicate that we wish to change it */
	if (!MidiLock(MIDILOCK_MODIFY, MIDILOCK_WAITFOREVER))
	{
		/* If loading a bank or program, get the first instrument */
		if (buffer[0] > 1)
		{
			insName = 0;
			if (MidiNextInstrument(&insName) == 0xFFFF)
			{
				_cputs("ERROR: No Instruments in database!\r\n");
				MidiUnlock();
				exit(-1);
			}
			pgmName = insName;

			/* If loading a program, get the first bank */
			if (buffer[0] > 2)
			{
				bankName = 0;
				if (MidiNextBank(&bankName, insName) == 0xFFFFFFFF)
				{
					_cputs("ERROR: No Banks in first Instrument!\r\n");
					MidiUnlock();
					exit(-2);
				}
				pgmName = bankName;
			}
		}

		/* Load the Instrument/Bank/Program from the specified filename */
		if ((loop = MidiLoadDat(argv[1], pgmName, buffer[0])))
		{
			_cputs("ERROR: ");
			_cputs(MidiGenError(&buffer[0], 120, loop));
			_cputs("\r\n");
			MidiUnlock();
			exit(-5);
		}
	
		/* Unlock the database after changes. Indicate that we've finished changing it */
		MidiUnlock();
	}

	/* ******************* Now print out the GENMIDI.DLL database **************** */

	/* Indicate that we're accessing the database, but don't lock out other programs */
	if (!MidiLock(MIDILOCK_ACCESS, MIDILOCK_WAITFOREVER))
	{
		/* Start with first instrument */
		insName = 0;

		/* Get next Instrument name */
		while (MidiNextInstrument(&insName) != 0xFFFF)
		{
			/* Print the instrument name */
			_cputs("===============================================================\r\n");
			_cputs(insName);
			_cputs("\r\n");

			check_pause(3);

			/* Start with first bank */
			bankName = 0;

			/* Get next Bank name */
			while (MidiNextBank(&bankName, insName) != 0xFFFFFFFF)
			{
				/* Print the Bank name */
				_cputs("\t-------------------------------------------------------\r\n\t");
				_cputs(bankName);
				_cputs("\r\n\t-------------------------------------------------------\r\n");

				check_pause(3);

				pgmStruct = 0;

				/* Get the next program name */
				while (MidiNextPgm(&pgmName, bankName, &pgmStruct) != 0xFF)
				{
					/* Print the program name */
					_cputs("\t\t");
					_cputs(pgmName);
					_cputs("\r\n");

					check_pause(1);

					noteStruct = 0;
					/* Get the next note name */
					while ((noteName = MidiNextNote(&noteStruct, pgmName)))
					{
						/* Print the note name */
						_cputs("\t\t\t");
						_cputs(noteName);
						_cputs("\r\n");

						check_pause(1);
					}
				}
			}
		}

		/* Indicate that we're done accessing the database */
		MidiUnlock();
	}

	exit(0);
}
