/* AddPgms.c
 *
 * An example of using GENMIDI.DLL.
 *
 * This example shows how to call the DLL functions MidiAddInstrument(),
 * MidiAddBank(), MidiAddPgm(). and MidiAddNote() to add
 * an Instrument, Bank, Program and/or Note to the DLL's loaded database.
 *
 * It also demonstrates how to traverse the entire DLL database, listing
 * each instrument with its banks, programs, and note names.
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "..\genmidi.h"

long	lines = 0;
long	count = 1;


/************************** check_pause() ****************************
 * Check if we have filled the screen. If so, pause output waiting for
 * the user to press a key to continue.
 *********************************************************************/

void check_pause(long more)
{
	/* User wants the pause feature (ie, lines is not 0), and we filled the screen? */
	count += more;
	if (lines && (count >= lines))
	{
		count = 1;
		_cputs("--- Press ENTER to continue ---");
		getchar();
		_cputs("\r\n");
	}
}



/***************************** main() ********************************/

void main(int argc, char *argv[])
{
	unsigned char *	addArray[4];
	unsigned char *	ptr;
	unsigned char *	noteName;
	unsigned long	loop;
	unsigned char	i, pgmNum, insNum;
	unsigned short	bankNum;
	unsigned char	buffer[120];

	/* Assume we won't create/modify any Instrument, Bank, Program, or Note */
	addArray[0] = addArray[1] = addArray[2] = addArray[3] = 0;
	i = 0;

	/* Parse command line args */
	for (loop=1; loop < (unsigned long)argc; loop++)
	{
		/* An option? */
		ptr = argv[loop];
		if (*ptr == '/')
		{
			ptr++;
			switch ((*ptr) & 0x5F)
			{
				/* Screen lines */
				case 'L':
					lines = atol(ptr+1);
			}
		}

		/* Instrument, Bank, Program, or Note name to create/modify */
		else if (i < 4)
		{
			addArray[i] = ptr;
			i++;
		}
	}

	/* Lock the database before changes. Indicate that we wish to change it */
	if (!MidiLock(MIDILOCK_MODIFY, MIDILOCK_WAITFOREVER))
	{
		/* An Instrument name supplied? */
		if (addArray[0])
		{
			/* Check if this instrument name is already in the database, and get its number */
			insNum = (unsigned char)MidiGetInstrumentNum(addArray[0]);

			/* If Instrument wasn't found then create a new instrument */
			if (insNum == 0xFF)
			{
				/* Create instrument */
				if ((loop = MidiAddInstrument(0, addArray[0], 0x00)))
				{
					_cputs("ERROR: ");
					_cputs(MidiGenError(&buffer[0], 120, loop));
					_cputs("\r\n");
					MidiUnlock();
					exit(-1);
				}

				/* Get new instrument's number */
				insNum = (unsigned char)MidiGetInstrumentNum(MidiNames.Instrument);
			}

			/* Save the pointer set up by DLL for use with MidiAddBank() and MidiAddPgm() */
			addArray[0] = MidiNames.Instrument;
		}

		/* No Instrument name supplied, so just get the first instrument if a bank name was also
			supplied (ie, we're going to add the bank to the first instrument) */
		else if (addArray[1])
		{
			if ((insNum = (unsigned char)MidiNextInstrument(&addArray[0])) == 0xFF)
			{
				_cputs("ERROR: No Instruments in database!\r\n");
				MidiUnlock();
				exit(-2);
			}
		}

		/* A bank name supplied? */
		if (addArray[1])
		{
			/* Check if this bank name is already in the instrument, and get its number */
			bankNum = (unsigned short)MidiGetBankNum(addArray[1], insNum);

			/* If bank wasn't found then create a new bank (using dummy bank #) */
			if (bankNum == 0xFFFF)
			{
				bankNum = MidiDummyBankNum(addArray[0]);

				/* If a program name wasn't supplied, let's set MIDI_GM_BANK just for demonstration */
				i = 0x00;
				if (!addArray[2]) i = MIDI_GM_BANK;

				if ((loop = MidiAddBank(addArray[0], 0, addArray[1], bankNum, i)))
				{
					_cputs("ERROR: ");
					_cputs(MidiGenError(&buffer[0], 120, loop));
					_cputs("\r\n");
					MidiUnlock();
					exit(-3);
				}
			}

			/* Save the pointer set up by DLL for use with MidiAddPgm() */
			addArray[1] = MidiNames.Bank;
		}

		/* No Bank name supplied, so just get the first bank in this instrument if a program
			name is also supplied (ie, we'll add the program to the first bank) */
		else if (addArray[2])
		{
			if ((bankNum = (unsigned short)MidiNextBank(&addArray[1], addArray[0])) == 0xFFFF)
			{
				_cputs("ERROR: No Banks in first Instrument!\r\n");
				MidiUnlock();
				exit(-4);
			}
		}

		/* A program name supplied? */
		if (addArray[2])
		{
			/* Check if this program name is already in the bank, and get its number */
			pgmNum = MidiGetPgmNum(addArray[2], bankNum, insNum);

			/* If program wasn't found then create/modify program #0 */
			if (pgmNum == 0xFF)
			{
				/* Create/Modify program number 0 */
				if ((loop = MidiAddPgm(addArray[0], addArray[1], addArray[2], 0, 0x00)))
				{
					_cputs("ERROR: ");
					_cputs(MidiGenError(&buffer[0], 120, loop));
					_cputs("\r\n");
					MidiUnlock();
					exit(-5);
				}
			}

			/* Save the pointer set up by DLL for use with MidiAddNote() */
			addArray[2] = MidiNames.Pgm;
		}

		/* No program name supplied, so just get the first program in this bank if note name was also
		   supplied */
		else if (addArray[3])
		{
			ptr = 0;
			if (MidiNextPgm(&addArray[2], addArray[1], &ptr) == 0xFF)
			{
				_cputs("ERROR: No Programs in first Bank!\r\n");
				MidiUnlock();
				exit(-6);
			}
		}

		/* A Note name supplied? */
		if (addArray[3])
		{
			/* Check if this note name is already in the program, and get its number. If not found, then
			create/modify note #0 */
			if((i = MidiGetNoteNum(addArray[3], addArray[2], &i)) > 127) i = 0;

			/* Create/Modify note */
			if ((loop = MidiAddNote(addArray[0], addArray[1], addArray[2], addArray[3], i, 0x00)))
			{
				_cputs("ERROR: ");
				_cputs(MidiGenError(&buffer[0], 120, loop));
				_cputs("\r\n");
				MidiUnlock();
				exit(-7);
			}
		}

		/* Unlock the database after changes. Indicate that we've finished changing it */
		MidiUnlock();
	}

	/* ******************* Now print out the GENMIDI.DLL database **************** */

	/* Indicate that we're accessing the database, but don't lock out other programs */
	if (!MidiLock(MIDILOCK_ACCESS, MIDILOCK_WAITFOREVER))
	{
		/* Start with first instrument */
		addArray[0] = 0;

		/* Get next Instrument name */
		while (MidiNextInstrument(&addArray[0]) != 0xFFFF)
		{
			/* Print the instrument name */
			_cputs("===============================================================\r\n");
			_cputs(addArray[0]);
			_cputs("\r\n");

			check_pause(3);

			/* Start with first bank */
			addArray[1] = 0;

			/* Get next Bank name */
			while (MidiNextBank(&addArray[1], addArray[0]) != 0xFFFFFFFF)
			{
				/* Print the Bank name */
				_cputs("\t-------------------------------------------------------\r\n\t");
				_cputs(addArray[1]);
				_cputs("\r\n\t-------------------------------------------------------\r\n");

				check_pause(3);

				addArray[2] = 0;

				/* Get the next program name */
				while (MidiNextPgm(&ptr, addArray[1], &addArray[2]) != 0xFF)
				{
					/* Print the program name */
					_cputs("\t\t");
					_cputs(ptr);
					_cputs("\r\n");

					check_pause(1);

					addArray[3] = 0;
					/* Get the next note name */
					while ((noteName = MidiNextNote(&addArray[3], ptr)))
					{
						/* Print the note name */
						_cputs("\t\t\t");
						_cputs(noteName);
						_cputs("\r\n");

						check_pause(1);
					}
				}
			}
		}

		/* Indicate that we're done accessing the database */
		MidiUnlock();
	}

	exit(0);
}
